
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <SDL/SDL.h>

#include "formats.h"
#include "viewer.h"
#include "main.h"

int init_viewer(void)
{
  if (SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK) < 0) {
    fprintf(stderr, "Couldn't initialize SDL: %s\n", SDL_GetError());
    exit(1);
  }
  atexit(SDL_Quit); /* Clean up on exit */

  viewer.framebuffermutex = SDL_CreateMutex();
  viewer.screen = NULL;
  viewer.buffer = NULL;
  viewer.previewing = 0;
  viewer.viewer_thread = SDL_CreateThread(viewer_thread, NULL);

  return 0;
}

int end_viewer(void)
{
  SDL_Event quitevent;

  viewer.previewing = 0;
  quitevent.type = SDL_USEREVENT;
  quitevent.user.code = SDL_USEREVENT_QUIT;
  SDL_PushEvent(&quitevent);
  SDL_WaitThread(viewer.viewer_thread, NULL);

  SDL_DestroyMutex(viewer.framebuffermutex);

  return 0;
}

int setup_sdl_window(int width, int height, int bpp)
{
  int sdl_flags = SDL_SWSURFACE | SDL_DOUBLEBUF;

  SDL_LockMutex(viewer.framebuffermutex);

  fprintf(stderr, "Setting up window; w = %d, h = %d, bpp = %d\n", width, height, bpp);

  if (width == 0 || height == 0) {
    SDL_UnlockMutex(viewer.framebuffermutex);
    return -1;
  }

/*
  if (SDL_InitSubSystem(SDL_INIT_VIDEO) < 0) {
    fprintf(stderr, "Couldn't initialize SDL video: %s\n", SDL_GetError());
    exit(1);
  }
*/

  viewer.screen = SDL_SetVideoMode(width, height, bpp, sdl_flags);
  if (viewer.screen == NULL) {
    fprintf(stderr, "Couldn't set SDL mode: %s\n", SDL_GetError());
    exit(1);
  }

  viewer.buffer = malloc(height*width*bpp/8);
  
  /* Set the window manager title bar */
  SDL_WM_SetCaption("Viewer", "Viewer");

  /* Black screen */
  SDL_FillRect(viewer.screen,NULL,0);
  SDL_UpdateRect(viewer.screen,0,0,0,0);

  viewer.width = width;
  viewer.height = height;
  viewer.bpp = bpp;
  viewer.previewing = 1;
  SDL_UnlockMutex(viewer.framebuffermutex);

  return 0;
}

int close_sdl_window(void)
{
  SDL_LockMutex(viewer.framebuffermutex);
//  SDL_QuitSubSystem(SDL_INIT_VIDEO);
  viewer.screen = NULL;
  viewer.previewing = 0;
  if (viewer.buffer != NULL)
    free(viewer.buffer);
  viewer.buffer = NULL;
  SDL_UnlockMutex(viewer.framebuffermutex);
  return 0;
}

int viewer_thread(void *data)
{
  int ret;
 
  viewer.screen = NULL;

  ret = sdl_event_thread();

  return ret;
}

int sdl_event_thread(void)
{
  SDL_Event event;
  int ret;

  while(1) {
    ret = SDL_WaitEvent(&event);
    if (ret != 1) {
      fprintf(stderr, "waitevent returned %d\n", ret);
      fprintf(stderr, "  %s\n", SDL_GetError());
      exit(1);
    }
//    fprintf(stderr, "Got event; type = %d\n", event.type);
    switch(event.type){
      case SDL_KEYDOWN:
        switch(event.key.keysym.sym){
          default:
            break;
        }
        break;
      case SDL_QUIT:
        fprintf(stderr, "Got quit\n");
        active = 0;
        break;
      case SDL_USEREVENT:
        switch (event.user.code) {
          case SDL_USEREVENT_STARTPREVIEW:
            fprintf(stderr, "Got start preview event\n");
            setup_sdl_window((long)event.user.data1, (long)event.user.data2, 24);
            break;
          case SDL_USEREVENT_ENDPREVIEW:
//            fprintf(stderr, "Got stop preview event\n");
            close_sdl_window();
            break;
          case SDL_USEREVENT_QUIT:
            return 0;
            break;
          case SDL_USEREVENT_NEWIMAGE:
            sdl_update_image(viewer.screen);
            break;
          default:
            break;
        }
        break;
    }
  }

  return 0;
}

void sdl_update_image()
{
  int framesize = viewer.width*viewer.height*palette_data[PALETTE_RGB24].pixeldepth;

  SDL_LockMutex(viewer.framebuffermutex);

  if (!viewer.previewing) {
    SDL_UnlockMutex(viewer.framebuffermutex);
    return;
  }

  memmove(viewer.screen->pixels, viewer.buffer, framesize);

  SDL_UpdateRect(viewer.screen,0,0,0,0);

  SDL_UnlockMutex(viewer.framebuffermutex);
}

int start_preview(int width, int height)
{
  SDL_Event event;

  event.type = SDL_USEREVENT;
  event.user.code = SDL_USEREVENT_STARTPREVIEW;
  event.user.data1 = (void*)width;
  event.user.data2 = (void*)height;

  SDL_PushEvent(&event);

  return 0;
}

int end_preview(void)
{
  SDL_Event event;

  event.type = SDL_USEREVENT;
  event.user.code = SDL_USEREVENT_ENDPREVIEW;
  SDL_PushEvent(&event);

  return 0;
}

int update_preview()
{
  sdl_update_image();

  return 0;
}

