
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include "oflow_messages.h"

int unpack_oflow_message(oflow_message_t *outmessage, void *indata)
{
  unsigned long checksum = 0;
  int i;
  unsigned long *data = (unsigned long *)indata;
  
  if (outmessage == NULL || indata == NULL)
    return -1;

  outmessage->header.framenum = ntohl(data[0]);
  outmessage->header.startelement = ntohl(data[1]);
  outmessage->header.nelements = ntohl(data[2]);

  if (outmessage->header.nelements > VELS_PER_MESSAGE)
    return -1;
  
  outmessage->header.checksum = ntohl(data[3]);

  checksum += outmessage->header.framenum;
  checksum += outmessage->header.startelement;
  checksum += outmessage->header.nelements;
  checksum += outmessage->header.checksum;
  
  for (i = 0; i < outmessage->header.nelements; i++) {
    outmessage->data.vel[2*i] = ntohl(data[4+2*i]);
    checksum += outmessage->data.vel[2*i];
    outmessage->data.vel[2*i+1] = ntohl(data[4+2*i+1]);
    checksum += outmessage->data.vel[2*i+1];
  }

  if (checksum != 0) {
    return -1;
  }

  return 0;
}

int pack_oflow_message(void *outdata, oflow_message_t *inmessage)
{
  unsigned long checksum = 0;
  int i;
  unsigned long *data = (unsigned long *)outdata;
  
  if (inmessage == NULL || outdata == NULL)
    return -1;

  if (inmessage->header.nelements > VELS_PER_MESSAGE)
    return -1;

  data[0] = htonl(inmessage->header.framenum);
  data[1] = htonl(inmessage->header.startelement);
  data[2] = htonl(inmessage->header.nelements);

  checksum += inmessage->header.framenum;
  checksum += inmessage->header.startelement;
  checksum += inmessage->header.nelements;
  
  for (i = 0; i < inmessage->header.nelements; i++) {
    data[4+2*i] = htonl(inmessage->data.vel[2*i]);
    checksum += inmessage->data.vel[2*i];
    data[4+2*i+1] = htonl(inmessage->data.vel[2*i+1]);
    checksum += inmessage->data.vel[2*i+1];
  }

  checksum = ~checksum + 1;
  data[3] = htonl(checksum);

  return 4*(4+2*inmessage->header.nelements);
}

int oflow_udp_open(char *remoteip, int remoteport, int localport)
{
  struct sockaddr_in client, server;
  int sfd = -1;
  
  if (inet_aton(remoteip, &server.sin_addr) == 0) {
    fprintf(stderr, "Invalid remote IP specified\n");
    goto error;
  }
  server.sin_family = AF_INET;
  server.sin_port = htons(localport);

  sfd = socket(PF_INET, SOCK_DGRAM, 0);
  if (sfd == -1) {
    fprintf(stderr, "Unable to create socket: %s\n", strerror(errno));
    goto error;
  }
  
  client.sin_addr.s_addr = INADDR_ANY;
  client.sin_family = AF_INET;
  client.sin_port = htons(localport);

  if (bind(sfd, (struct sockaddr *)&client, sizeof(client)) == -1) {
    fprintf(stderr, "Unable to bind to local port: %s\n", strerror(errno));
    goto error;
  }
  
  if (connect(sfd, (struct sockaddr *)&server, sizeof(server)) == -1) {
    fprintf(stderr, "Unable to connect to remote host: %s\n", strerror(errno));
    goto error;
  }

  return sfd;

 error:
  if (sfd != -1)
    close(sfd);
  return -1;
}

int oflow_udp_close(int sfd)
{
  return close(sfd);
}

int oflow_udp_send(int sfd, void *data, int length)
{
  int ret;

  ret = send(sfd, data, length, 0);
  if (ret == -1) {
    return -errno;
  }
  
  return 0;
}

int oflow_udp_recv(int sfd, void *data, int length)
{
  int ret;

  ret = recv(sfd, data, length, MSG_TRUNC);
  if (ret == -1) {
    return -errno;
  }

  return ret;
}


