
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>

#include <xmmintrin.h>
#include <opencv/cv.h>

#include "oflow.h"
#include "formats.h"
#include "imgconvert.h"
#include "oflow_messages.h"

int oflow_init(oflow_state_t *state)
{
  oflow_options_t *options = &state->options;
  int depth = palette_data[options->palette].pixeldepth;
  int h = options->height;
  int w = options->width;
  int ret;
 
  if (state == NULL)
    goto error;
 
  state->tmp_image = NULL;
  state->prev_image = NULL;
  state->tmp_header = NULL;
  state->prev_header = NULL;
  state->preview_header = NULL;
  state->velx = NULL;
  state->vely = NULL;
  state->velx_header = NULL;
  state->vely_header = NULL;
  state->velx_avg = NULL;
  state->vely_avg = NULL;
  state->send_data = NULL;
  state->send_msg = NULL;
  
  state->tmp_image = (unsigned char*)malloc(h*w*depth);
  if (state->tmp_image == NULL)
    goto error;

  state->prev_image = (unsigned char*)malloc(h*w*depth);
  if (state->prev_image == NULL)
    goto error;

  state->tmp_header = cvCreateImageHeader(cvSize(w,h), IPL_DEPTH_8U, 1);
  if (state->tmp_header == NULL)
    goto error;
  cvSetData(state->tmp_header, state->tmp_image, w);

  state->prev_header = cvCreateImageHeader(cvSize(w,h), IPL_DEPTH_8U, 1);
  if (state->prev_header == NULL)
    goto error;
  cvSetData(state->prev_header, state->prev_image, w);

  state->preview_header = cvCreateImageHeader(cvSize(w,h), IPL_DEPTH_8U, 3);
  if (state->preview_header == NULL)
    goto error;
  cvSetData(state->preview_header, state->viewer->buffer, 3*w);

  state->velx = (float*)malloc(h*w*sizeof(float));
  if (state->velx == NULL)
    goto error;
  
  state->vely = (float*)malloc(h*w*sizeof(float));
  if (state->vely == NULL)
    goto error;

  state->velx_header = cvCreateImageHeader(cvSize(w,h), IPL_DEPTH_32F, 1);
  if (state->velx_header == NULL)
    goto error;
  cvSetData(state->velx_header, state->velx, w*sizeof(float));

  state->vely_header = cvCreateImageHeader(cvSize(w,h), IPL_DEPTH_32F, 1);
  if (state->vely_header == NULL)
    goto error;
  cvSetData(state->vely_header, state->vely, w*sizeof(float));
  
  state->velx_avg = (float*)malloc((h/options->avgwindowsize)*(w/options->avgwindowsize)*sizeof(float));
  if (state->velx_avg == NULL)
    goto error;
  
  state->vely_avg = (float*)malloc((h/options->avgwindowsize)*(w/options->avgwindowsize)*sizeof(float));
  if (state->vely_avg == NULL)
    goto error;

  state->capture = cvCaptureFromCAM(atoi(options->device));
  if (state->capture == NULL) {
    fprintf(stderr, "Unable to open capture device %s (video%d)\n", options->device, atoi(options->device));
    goto error;
  }

  ret = cvSetCaptureProperty(state->capture, CV_CAP_PROP_FRAME_WIDTH, (double)options->width);
  if (ret != 0) {
    fprintf(stderr, "Unable to set capture width\n");
    goto error;
  }
  if (cvSetCaptureProperty(state->capture, CV_CAP_PROP_FRAME_HEIGHT, (double)options->height) != 0) {
    fprintf(stderr, "Unable to set capture height\n");
    goto error;
  }
/*
  if (cvSetCaptureProperty(state->capture, CV_CAP_PROP_FPS, (double)options->fps) != 0) {
    fprintf(stderr, "Unable to set capture fps\n");
    goto error;
  }
*/
  if (cvQueryFrame(state->capture) == NULL) {
    fprintf(stderr, "Unable to read a frame from capture device %s\n", options->device);
    goto error;
  }

  state->sfd = oflow_udp_open(options->remoteip, options->remoteport, options->localport);
  if (state->sfd < 0) {
    fprintf(stderr, "Error connecting to remote host\n");
    goto error;
  }

  state->send_data = malloc(OFLOW_MESSAGE_SIZE);
  if (state->send_data == NULL) {
    goto error;
  }

  state->send_msg = malloc(OFLOW_MESSAGE_SIZE);
  if (state->send_msg == NULL) {
    goto error;
  }

  state->initialized = 0;

  return 0;

 error:
  if (state != NULL) {
    if (state->tmp_image != NULL)
      free(state->tmp_image);
    if (state->prev_image != NULL)
      free(state->prev_image);
    if (state->tmp_header != NULL)
      cvReleaseImageHeader((IplImage**)&state->tmp_header);
    if (state->prev_header != NULL)
      cvReleaseImageHeader((IplImage**)&state->prev_header);
    if (state->preview_header != NULL)
      cvReleaseImageHeader((IplImage**)&state->preview_header);
    if (state->velx != NULL)
      free(state->velx);
    if (state->vely != NULL)
      free(state->vely);
    if (state->velx_header != NULL)
      cvReleaseImageHeader((IplImage**)&state->velx_header);
    if (state->vely_header != NULL)
      cvReleaseImageHeader((IplImage**)&state->vely_header);
    if (state->send_data != NULL)
      free(state->send_data);
    if (state->send_msg != NULL)
      free(state->send_msg);
  }
  return -1;
}

int oflow_start(oflow_state_t *state)
{
//  oflow_options_t *options = &state->options;
//  int h = options->height;
//  int w = options->width;

  state->fps = 0;
  state->framenum = 0;
  state->framenum_prev = 0;
  gettimeofday(&state->tv_fps, NULL);

  state->udp_errors = 0;
  
  state->initialized = 0;

  return 0;
}

int oflow_process(oflow_state_t *state)
{
  oflow_options_t *options = &state->options;
  int h = options->height;
  int w = options->width;
  unsigned char *tmp_image;
  CvArr* tmp_header;
  float velx_avg, vely_avg;
  float velx_fullavg, vely_fullavg;
  float velx_max, vely_max;
  int x, y, xx, yy, x2, y2;
  struct timeval tv_initial, tv_start, tv_end, tv_final;
  int avgwindowsize = options->avgwindowsize;
  IplImage *frame;
  float fps_time;
  int ret = 0;
  int retval;
  float full_time;

  gettimeofday(&tv_start, NULL);
  tv_initial = tv_start;

  frame = cvQueryFrame(state->capture);
  if (frame == NULL)
    return -1;

  cvConvertImage(frame, state->tmp_header, 0);

  gettimeofday(&tv_end, NULL);
  state->acquiredur = 1000.0*(tv_end.tv_sec-tv_start.tv_sec) + 0.001*(tv_end.tv_usec-tv_start.tv_usec);
  tv_start = tv_end;

  if (state->viewer->previewing) {
    cvSetData(state->preview_header, state->viewer->buffer, 3*w);
    cvConvertImage(frame, state->preview_header, 0);
//    imgconvert((char*)state->fg.current_image, options->palette, (char*)state->viewer->buffer, PALETTE_RGB24, w, h);
    update_preview();
  }

  gettimeofday(&tv_end, NULL);
  state->previewdur = 1000.0*(tv_end.tv_sec-tv_start.tv_sec) + 0.001*(tv_end.tv_usec-tv_start.tv_usec);
  tv_start = tv_end;
  
  if (!state->initialized) {
    tmp_image = state->tmp_image;
    state->tmp_image = state->prev_image;
    state->prev_image = tmp_image;
    tmp_header = state->tmp_header;
    state->tmp_header = state->prev_header;
    state->prev_header = tmp_header;
    state->initialized = 1;
    state->framenum++;
    return 0;
  }

  cvCalcOpticalFlowLK(state->prev_header, state->tmp_header, cvSize(options->windowsize,options->windowsize), state->velx_header, state->vely_header);

  velx_avg = vely_avg = 0;
  velx_fullavg = vely_fullavg = 0;
  velx_max = vely_max = 0;
 
  for (y = 0, y2 = 0; y < h; y += avgwindowsize, y2++) {
    for (x = 0, x2 = 0; x < w; x += avgwindowsize, x2++) {
      velx_avg = 0;
      vely_avg = 0;
      for (yy = 0; yy < avgwindowsize; yy++) {
        for (xx = 0; xx < avgwindowsize; xx++) {
          velx_avg += state->velx[w*(y+yy)+(x+xx)];
          vely_avg += state->vely[w*(y+yy)+(x+xx)];
          velx_fullavg += state->velx[w*(y+yy)+(x+xx)];
          vely_fullavg += state->vely[w*(y+yy)+(x+xx)];
          velx_max = MAX(fabs(state->velx[w*(y+yy)+(x+xx)]),velx_max);
          vely_max = MAX(fabs(state->vely[w*(y+yy)+(x+xx)]),vely_max);
	}
      }
      state->velx_avg[y2*(w/avgwindowsize)+x2] = velx_avg / (1.0*avgwindowsize*avgwindowsize);
      state->vely_avg[y2*(w/avgwindowsize)+x2] = vely_avg / (1.0*avgwindowsize*avgwindowsize);
    }
  }
  state->velx_fullavg = velx_fullavg / (1.0*w*h);
  state->vely_fullavg = vely_fullavg / (1.0*w*h);
  state->velx_max = velx_max;
  state->vely_max = vely_max;

  if ((retval = oflow_send_message(state)) < 0) {
    state->udp_errors++;
    ret = -2;
  }

  tmp_image = state->tmp_image;
  state->tmp_image = state->prev_image;
  state->prev_image = tmp_image;

  tmp_header = state->tmp_header;
  state->tmp_header = state->prev_header;
  state->prev_header = tmp_header;

  gettimeofday(&tv_end, NULL);
  state->processdur = 1000.0*(tv_end.tv_sec-tv_start.tv_sec) + 0.001*(tv_end.tv_usec-tv_start.tv_usec);

  state->framenum++;

  fps_time = 1000.0*(tv_end.tv_sec-state->tv_fps.tv_sec) + 0.001*(tv_end.tv_usec-state->tv_fps.tv_usec);
  if (fps_time >= 1000) {
    state->fps = 1000.0*(state->framenum-state->framenum_prev) / fps_time;
    state->framenum_prev = state->framenum;
    state->tv_fps = tv_end;
  }

  gettimeofday(&tv_final, NULL);
  full_time = 1000.0*(tv_final.tv_sec-tv_initial.tv_sec) + 0.001*(tv_final.tv_usec-tv_initial.tv_usec);

  if (options->maxfreq > 0) {
    float minperiod = 1000.0/options->maxfreq;
    if (minperiod > full_time) {
      usleep(1000*(unsigned long)(minperiod-full_time));
    }
  }

  return ret;
}

int oflow_stop(oflow_state_t *state)
{
//  oflow_options_t *options = &state->options;

  return 0;
}

int oflow_destroy(oflow_state_t *state)
{
//  oflow_options_t *options = &state->options;

  if (state->tmp_image != NULL) {
    free(state->tmp_image);
    state->tmp_image = NULL;
  }

  if (state->prev_image != NULL) {
    free(state->prev_image);
    state->prev_image = NULL;
  }

  cvReleaseImageHeader((IplImage**)&state->tmp_header);
  cvReleaseImageHeader((IplImage**)&state->prev_header);
  cvReleaseImageHeader((IplImage**)&state->preview_header);

  if (state->velx != NULL) {
    free(state->velx);
    state->velx = NULL;
  }

  if (state->vely != NULL) {
    state->vely = NULL;
    free(state->vely);
  }

  cvReleaseImageHeader((IplImage**)&state->velx_header);
  cvReleaseImageHeader((IplImage**)&state->vely_header);

  cvReleaseCapture(&state->capture);  

  if (state->send_data != NULL) {
    free(state->send_data);
    state->send_data = NULL;
  }

  if (state->send_msg != NULL) {
    free(state->send_msg);
    state->send_msg = NULL;
  }

  if (state->sfd >= 0) {
    oflow_udp_close(state->sfd);
    state->sfd = -1;
  }

  return 0;
}

int oflow_send_message(oflow_state_t *state)
{
  int i, j;
  char *data = state->send_data;
  oflow_message_t *msg = state->send_msg;
  oflow_options_t *options = &state->options;
  int h = options->height;
  int w = options->width;
  int imgsize = (h/options->avgwindowsize)*(w/options->avgwindowsize);
  int ret = -1;
  int size;
  oflow_message_t tmp_msg;

  msg->header.framenum = state->framenum;

  for (i = 0; i < imgsize; i += VELS_PER_MESSAGE) {
    msg->header.startelement = i;
    msg->header.nelements = MIN(VELS_PER_MESSAGE,imgsize-i);
    for (j = 0; j < msg->header.nelements; j++) {
      msg->data.vel[2*j] = (unsigned long)(long)(state->velx_avg[i+j]*100.0);
      msg->data.vel[2*j+1] = (unsigned long)(long)(state->vely_avg[i+j]*100.0);
    }
    if ((size = pack_oflow_message((void*)data, msg)) < 0) {
      fprintf(stderr, "Error packing output message\n");
      goto error;
    }
    if (unpack_oflow_message(&tmp_msg, (void*)data) < 0) {
      fprintf(stderr, "Error unpacking output message\n");
      goto error;
    }
    if ((ret = oflow_udp_send(state->sfd, data, size)) < 0) {
      fprintf(stderr, "Error sending packet: %s\n", strerror(-ret));
      goto error;
    }
  }

  return 0;

 error:
  return ret;
}

