
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <pthread.h>
#include <getopt.h>

#include "main.h"
#include "oflow.h"
#include "cfg.h"
#include "oflow_messages.h"

viewerstatus_t viewer;

oflow_state_t oflow;

int active = 1;

void *oflow_thread(void *data);

void sigterm_handler(int signum)
{
  fprintf(stderr, "Quit signal received, quitting...\n");
  active = 0;
}

int main(int argc, char **argv)
{
  char c;
  int preview = 0;
  struct sigaction sigactiondata;
  pthread_attr_t attrs;
  pthread_t oflow_tid;

  while ((c = getopt(argc, argv, "p")) != EOF) {
    switch (c) {
      case 'p':
        preview = 1;
        break;
      case '?':
      default:
        fprintf(stderr, "Unknown option.\n");
        return 1;
        break;
    }
  }
 
  argc -= optind-1;

  
  if (argc != 2) {
    printf("Usage: %s [options] <cfg file>\n", argv[0]);
    return 1;
  }

  if (parse_cfg(argv[optind], &oflow.options) < 0) {
    return -1;
  }

  oflow.viewer = &viewer;
  
  init_viewer();

  if (oflow_init(&oflow) < 0) {
    return -1;
  }

  if (preview) {
    start_preview(oflow.options.width, oflow.options.height);
  }

  if (oflow_start(&oflow) < 0) {
    return 1;
  }

  sigactiondata.sa_handler = sigterm_handler;
  sigactiondata.sa_sigaction = 0;
  sigactiondata.sa_flags = 0;
  sigaction(SIGQUIT, &sigactiondata, NULL);

  pthread_attr_init(&attrs);
  pthread_attr_setdetachstate(&attrs, PTHREAD_CREATE_DETACHED);

  if (pthread_create(&oflow_tid, &attrs, oflow_thread, NULL) != 0) {
    fprintf(stderr, "Unable to start oflow thread\n");
    goto done;
  }
  
  while (active) {
    sleep(1);
    printf("\n");
    printf("Frame: %d  UDP Errors: %d\n", oflow.framenum, oflow.udp_errors);
    printf("fps = %0.2f\n", oflow.fps);
    printf("avg: velx = %0.2f, vely = %0.2f\n", oflow.velx_fullavg, oflow.vely_fullavg);
    printf("max: velx = %0.2f, vely = %0.2f\n", oflow.velx_max, oflow.vely_max);
  }

  printf("Finished...\n");

 done:
  
  oflow_stop(&oflow);

  if (preview) {
    end_preview();
  }
  
  end_viewer();

  oflow_destroy(&oflow);
  
  return 0;
}

void *oflow_thread(void *data)
{

  printf("Starting loop\n");

  while (active) {
    if (oflow_process(&oflow) == -1) {
      active = 0;
    }
  }

  return NULL;
}

