
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "oflow.h"

int parse_cfg(char *filename, oflow_options_t *options)
{
  FILE *file;
  int linenum;
  char line[256];
  char *lineptr, *lineptr2;
  int tmpi;

  file = fopen(filename, "r");
  if (file == NULL) {
    fprintf(stderr, "Unable to open config file %s: %s\n", filename, strerror(errno));
    goto error;
  }

  linenum = 0;
  while (fgets(line, sizeof(line), file) != NULL) { 
    linenum++;
    line[strcspn(line, "\n\r")] = '\0';
    if (line[0] == '\0' || line[0] == '#')
      continue;
    lineptr = strtok(line, "=");
    lineptr2 = strtok(NULL, "=");
    if (lineptr == NULL || lineptr2 == NULL) {
      fprintf(stderr, "Unable to parse line %d of config file %s\n", linenum, filename);
      goto error;
    }

    if (strcasecmp(lineptr, "device") == 0) {
      strncpy(options->device, lineptr2, 256);
    }
    else if (strcasecmp(lineptr, "width") == 0) {
      tmpi = atoi(lineptr2);
      if (tmpi <= 0 || tmpi > 4096) {
        fprintf(stderr, "Invalid width on line %d of config file %s\n", linenum, filename);
        goto error;
      }
      options->width = tmpi;
    }
    else if (strcasecmp(lineptr, "height") == 0) {
      tmpi = atoi(lineptr2);
      if (tmpi <= 0 || tmpi > 4096) {
        fprintf(stderr, "Invalid height on line %d of config file %s\n", linenum, filename);
        goto error;
      }
      options->height = tmpi;
    }
    else if (strcasecmp(lineptr, "palette") == 0) {
      if (strcasecmp(lineptr2, "rgb24") == 0) {
        options->palette = PALETTE_RGB24;
      }
      else if (strcasecmp(lineptr2, "yuv420p") == 0) {
        options->palette = PALETTE_YUV420P;
      }
      else if (strcasecmp(lineptr2, "yuv422p") == 0) {
        options->palette = PALETTE_YUV422P;
      }
      else if (strcasecmp(lineptr2, "gray8") == 0) {
        options->palette = PALETTE_GRAY8;
      }
      else {
        fprintf(stderr, "Invalid palette on line %d of config file %s\n", linenum, filename);
        goto error;
      }
    }
    else if (strcasecmp(lineptr, "windowsize") == 0) {
      tmpi = atoi(lineptr2);
      if (tmpi <= 0 || tmpi > 4096) {
        fprintf(stderr, "Invalid window size on line %d of config file %s\n", linenum, filename);
        goto error;
      }
      options->windowsize = tmpi;
    }
    else if (strcasecmp(lineptr, "avgwindowsize") == 0) {
      tmpi = atoi(lineptr2);
      if (tmpi <= 0 || tmpi > 4096) {
        fprintf(stderr, "Invalid averaging window size on line %d of config file %s\n", linenum, filename);
        goto error;
      }
      options->avgwindowsize = tmpi;
    }
    else if (strcasecmp(lineptr, "maxfreq") == 0) {
      tmpi = atoi(lineptr2);
      if (tmpi < 0 || tmpi > 1000) {
        fprintf(stderr, "Invalid maximum frequency on line %d of config file %s\n", linenum, filename);
        goto error;
      }
      options->maxfreq = tmpi;
    }
    else if (strcasecmp(lineptr, "remoteip") == 0) {
      strncpy(options->remoteip, lineptr2, 16);
    }
    else if (strcasecmp(lineptr, "remoteport") == 0) {
      tmpi = atoi(lineptr2);
      if (tmpi <= 0 || tmpi > 65535) {
        fprintf(stderr, "Invalid remote port on line %d of config file %s\n", linenum, filename);
        goto error;
      }
      options->remoteport = tmpi;
    }
    else if (strcasecmp(lineptr, "localport") == 0) {
      tmpi = atoi(lineptr2);
      if (tmpi <= 0 || tmpi > 65535) {
        fprintf(stderr, "Invalid local port on line %d of config file %s\n", linenum, filename);
        goto error;
      }
      options->localport = tmpi;
    }
    else {
      fprintf(stderr, "Unknown option \"%s\" on line %d of config file %s\n", lineptr, linenum, filename);
      goto error;
    }
  }

  fclose(file);
  
  return 0;

 error:
  if (file != NULL)
    fclose(file);
  return -1;
}

