#include "GL/openglut.h"
#include "GL/glext.h"

#include <assert.h>
#include <stdio.h>

PFNGLUSEPROGRAMPROC					glUseProgram = 0;
PFNGLGENVERTEXARRAYSPROC			glGenVertexArrays = 0;
PFNGLBINDVERTEXARRAYPROC			glBindVertexArray = 0;
PFNGLGENBUFFERSPROC					glGenBuffers = 0;
PFNGLBINDBUFFERPROC					glBindBuffer = 0;
PFNGLBUFFERDATAPROC					glBufferData = 0;
PFNGLVERTEXATTRIBPOINTERPROC		glVertexAttribPointer = 0;
PFNGLENABLEVERTEXATTRIBARRAYPROC	glEnableVertexAttribArray = 0;
PFNGLCREATEPROGRAMPROC				glCreateProgram = 0;
PFNGLCREATESHADERPROC				glCreateShader = 0;
PFNGLSHADERSOURCEPROC				glShaderSource = 0;
PFNGLCOMPILESHADERPROC				glCompileShader = 0;
PFNGLGETSHADERIVPROC				glGetShaderiv = 0;
PFNGLGETSHADERINFOLOGPROC			glGetShaderInfoLog = 0;
PFNGLATTACHSHADERPROC				glAttachShader = 0;
PFNGLDELETESHADERPROC				glDeleteShader = 0;
PFNGLBINDATTRIBLOCATIONPROC			glBindAttribLocation = 0;
PFNGLLINKPROGRAMPROC				glLinkProgram = 0;
PFNGLGETPROGRAMIVPROC				glGetProgramiv = 0;
PFNGLGETPROGRAMINFOLOGPROC			glGetProgramInfoLog = 0;
PFNGLVALIDATEPROGRAMPROC			glValidateProgram = 0;

const char *vpSource =
"#version 150 core\n"
"\n"
"in vec4 in_Position;\n"
"in vec4 in_Color;\n"
"out vec4 vert_Color;\n"
"void main(void)\n"
"{\n"
"	const vec4 positions[4] = vec4[4](\n"
"		vec4(-1, -1, 0, 1),\n"
"		vec4( 1, -1, 0, 1),\n"
"		vec4(-1,  1, 0, 1),\n"
"		vec4( 1,  1, 0, 1)\n"
"	);\n"
"\n"
"	const vec4 colors[4] = vec4[4](\n"
"		vec4(1,0,0,1),\n"
"		vec4(0,1,0,1),\n"
"		vec4(1,0,1,1),\n"
"		vec4(1,1,0,1)\n"
"	);\n"
"\n"
"	// Uncomment one line from each group.\n"
"\n"
"	gl_Position = in_Position; // Read from VBO\n"
"	//gl_Position = positions[gl_VertexID]; // read from array\n"
"\n"
"	vert_Color = in_Color; // Read from VBO\n"
"	//vert_Color = colors[gl_VertexID]; // Read from array\n"
"	//vert_Color = vec4(0.25*gl_VertexID); // Generate procedurally\n"
"}\n";
const char *fpSource =
"#version 150 core\n"
"in vec4 vert_Color;\n"
"out vec4 out_Color;\n"
"void main(void)\n"
"{\n"
"    out_Color = vert_Color;\n"
"}\n";

GLuint prog_id = 0;
GLuint vao = 0;
GLuint vbo = 0;
static void init(void)
{
	// Create vertex array object
	glGenVertexArrays(1, &vao);
	glBindVertexArray(vao);

	// Create vertex buffer
	float vertices[] = 
	{
		-1, -1, 0, 1,		1, 0, 0, 1,
		 1, -1, 0, 1,		0, 1, 0, 1,
		-1,  1, 0, 1,		1, 0, 1, 1,
		 1,  1, 0, 1,		1, 1, 0, 1
	};
	glGenBuffers(1, &vbo);
	glBindBuffer(GL_ARRAY_BUFFER, vbo);
	glBufferData(GL_ARRAY_BUFFER, 4*8*sizeof(float), vertices, GL_STATIC_DRAW);
	const GLuint positionAttribIndex = 0;
	const GLuint colorAttribIndex = 1;
	glVertexAttribPointer(positionAttribIndex, 4, GL_FLOAT, GL_FALSE, 8*sizeof(float), 0);
	glVertexAttribPointer(colorAttribIndex, 4, GL_FLOAT, GL_FALSE, 8*sizeof(float), (void*)(4*sizeof(float)));
	glEnableVertexAttribArray(positionAttribIndex);
	glEnableVertexAttribArray(colorAttribIndex);

	glBindVertexArray(0);

	prog_id = glCreateProgram();
	// Vertex program
	GLuint vpShader = glCreateShader(GL_VERTEX_SHADER);
	assert(vpShader != 0);
	glShaderSource(vpShader, 1, &vpSource, NULL);
	glCompileShader(vpShader);
	int vpStatus = 0;
	glGetShaderiv(vpShader, GL_COMPILE_STATUS, &vpStatus);
	{
		char infoLog[8192];
		glGetShaderInfoLog(vpShader, 8192, NULL, infoLog);
		printf("vertex shader compile log:\n%s\n", infoLog);
	}
	assert(vpStatus == 1);
	glAttachShader(prog_id, vpShader);
	glDeleteShader(vpShader);
	// Fragment program
	GLuint fpShader = glCreateShader(GL_FRAGMENT_SHADER);
	assert(fpShader != 0);
	glShaderSource(fpShader, 1, &fpSource, NULL);
	glCompileShader(fpShader);
	int fpStatus = 0;
	glGetShaderiv(fpShader, GL_COMPILE_STATUS, &fpStatus);
	{
		char infoLog[8192];
		glGetShaderInfoLog(fpShader, 8192, NULL, infoLog);
		printf("fragment shader compile log:\n%s\n", infoLog);
	}
	assert(fpStatus == 1);
	glAttachShader(prog_id, fpShader);
	glDeleteShader(fpShader);
	// Attribute indices
	glBindAttribLocation(prog_id, positionAttribIndex, "in_Position");
	glBindAttribLocation(prog_id, colorAttribIndex, "in_Color");
	// Link the program object
	glLinkProgram(prog_id);
	int status = 0;
	glGetProgramiv(prog_id, GL_LINK_STATUS, &status);
	{
		char infoLog[8192];
		glGetProgramInfoLog(prog_id, 8192, NULL, infoLog);
		printf("program link log:\n%s\n", infoLog);
	}
	assert(status == 1);

	// cleanup
	glUseProgram(0);
	{
		GLenum errorCode = glGetError();
		assert(errorCode == GL_NO_ERROR);
	}
}


static void renderScene(void)
{
	glClearColor(1.0f, 0.0f, 1.0f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT);
	glDisable(GL_DEPTH_TEST);

	// set state
	glUseProgram(prog_id);
	glBindVertexArray(vao);

	// Validate before use
	static bool firstTime = true;
	if (firstTime)
	{
		firstTime = false;
		glValidateProgram(prog_id);
		GLint status = 0;
		glGetProgramiv(prog_id, GL_VALIDATE_STATUS, &status);
		{
			char infoLog[8192];
			glGetProgramInfoLog(prog_id, 8192, NULL, infoLog);
			printf("program validation log:\n%s\n", infoLog);
		}
		assert(status == 1);
	}

	// Draw full-screen quad, pre-projected into clip space.
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

	// cleanup
	glBindVertexArray(0);
	glUseProgram(0);
	{
		GLenum errorCode = glGetError();
		assert(errorCode == GL_NO_ERROR);
	}
	glFlush();
}


int main(int argc, char **argv) {
	glutInit(&argc, argv);
	glutInitDisplayMode(GLUT_DEPTH | GLUT_SINGLE | GLUT_RGBA);
	glutInitWindowPosition(100,100);
	glutInitWindowSize(320,320);
	glutCreateWindow("Basic GLUT Project");
	glutDisplayFunc(renderScene);

	// Get extension pointers
#if defined(_MSC_VER)
	glUseProgram = (PFNGLUSEPROGRAMPROC)wglGetProcAddress("glUseProgram");
	glGenVertexArrays = (PFNGLGENVERTEXARRAYSPROC)wglGetProcAddress("glGenVertexArrays");
	glBindVertexArray = (PFNGLBINDVERTEXARRAYPROC)wglGetProcAddress("glBindVertexArray");
	glGenBuffers = (PFNGLGENBUFFERSPROC)wglGetProcAddress("glGenBuffers");
	glBindBuffer = (PFNGLBINDBUFFERPROC)wglGetProcAddress("glBindBuffer");
	glBufferData = (PFNGLBUFFERDATAPROC)wglGetProcAddress("glBufferData");
	glVertexAttribPointer = (PFNGLVERTEXATTRIBPOINTERPROC)wglGetProcAddress("glVertexAttribPointer");	
	glEnableVertexAttribArray = (PFNGLENABLEVERTEXATTRIBARRAYPROC)wglGetProcAddress("glEnableVertexAttribArray");
	glCreateProgram = (PFNGLCREATEPROGRAMPROC)wglGetProcAddress("glCreateProgram");
	glCreateShader = (PFNGLCREATESHADERPROC)wglGetProcAddress("glCreateShader");
	glShaderSource = (PFNGLSHADERSOURCEPROC)wglGetProcAddress("glShaderSource");
	glCompileShader = (PFNGLCOMPILESHADERPROC)wglGetProcAddress("glCompileShader");
	glGetShaderiv = (PFNGLGETSHADERIVPROC)wglGetProcAddress("glGetShaderiv");
	glGetShaderInfoLog = (PFNGLGETSHADERINFOLOGPROC)wglGetProcAddress("glGetShaderInfoLog");
	glAttachShader = (PFNGLATTACHSHADERPROC)wglGetProcAddress("glAttachShader");
	glDeleteShader = (PFNGLDELETESHADERPROC)wglGetProcAddress("glDeleteShader");				
	glBindAttribLocation = (PFNGLBINDATTRIBLOCATIONPROC)wglGetProcAddress("glBindAttribLocation");
	glLinkProgram = (PFNGLLINKPROGRAMPROC)wglGetProcAddress("glLinkProgram");
	glGetProgramiv = (PFNGLGETPROGRAMIVPROC)wglGetProcAddress("glGetProgramiv");
	glGetProgramInfoLog = (PFNGLGETPROGRAMINFOLOGPROC)wglGetProcAddress("glGetProgramInfoLog");
	glValidateProgram = (PFNGLVALIDATEPROGRAMPROC)wglGetProcAddress("glValidateProgram");
#endif

	init();

	glutMainLoop();
	return 0;
}