///////////////////////////////////////////////////////////////
// DATE       : 15-5-97
// AUTHOR     : Wim Rijnders
// LAST_UPDATE: 1-7-97
// TITLE      : MFC Interface functions for befunge.
// PROJECT    : VISBEF
// VERSION    : 1
// PLATFORM   : WIN32, MFC.
//-------------------------------------------------------------
// DESCRIPTION
// ===========
//
// The stuff in this file serves as the interface between
// the befunge core and the windows MFC stuff. Basically,
// it's the glue between Befunge and Windows.
//-------------------------------------------------------------
// NOTES
// =====
//
// * Prototypes for the hook functions are in BEFCLASS.H.
//-------------------------------------------------------------
// HISTORY
// =======
//
// 15-5-97 Wim Rijnders v1: Created it.
///////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "resource.h"
#include "CharInput.h"
#include "NumInput.h"
#include "mainfrm.h"
#include "visbefdoc.h"
#include "visbefview.h"
#include "befmfc.h"


//static declaration(s)
static CString strInput;


///////////////////////////////////////////////////////////////
// Global functions for accessing various MFC objects
///////////////////////////////////////////////////////////////


inline CVisBefDoc *GetDoc()
{
	return (CVisBefDoc * ) ( ( CMainFrame * ) AfxGetMainWnd() )->GetActiveDocument();
}


inline CVisBefView *GetView()
{
   return (CVisBefView * ) ( ( CMainFrame * ) AfxGetMainWnd() )->GetActiveView();
}


inline CEdit *GetOutputWindow()
{
	return (CEdit *) ( ( CMainFrame * ) AfxGetMainWnd() ) ->m_wndOutput.GetDlgItem(IDC_EDIT1);
}


inline CListBox *GetStackWindow()
{
	CMainFrame * pMainWnd = ( CMainFrame * ) AfxGetMainWnd();
	if ( pMainWnd && IsWindow(pMainWnd->m_hWnd) )
		return (CListBox *) pMainWnd->m_wndStackBar.GetDlgItem(IDC_LIST1);
	else
		return NULL;
}


///////////////////////////////////////////////////////////////
// Helper Functions.
///////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////
// Clear the input string buffer
//
void ClearInput()
{
	strInput.Empty();
}

///////////////////////////////////////////////////////////////
// Hook functions for communicating with befunge.
///////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Get next input character; if none available, prompt
// user for more.
//
// Note: This is meant to be analogous to the working of 
//       stdin for console programs.
//
// Other Note: As in console apps, \n is part of input.
//
char GetInputChar()
{
   CVisBefDoc *pDoc = GetDoc();
	BOOL bWasanimating = pDoc->IsAnimating();

	if ( bWasanimating)
		pDoc->StopAnimating();

	while (!strInput.GetLength() )
	{
		CCharInput Dlg;

		Dlg.DoModal();
		strInput = Dlg.m_input;

		if ( strInput.GetLength() )
			strInput += CString("\n");
	}

	char outchar = ( (LPCSTR) strInput )[0];
	strInput = strInput.Mid(1);

	if ( bWasanimating)
		pDoc->StartAnimating();

	//OutputChar(outchar);
	return outchar;
}


//////////////////////////////////////////////////////////
// Prompt user for integer value.
//
long GetInputNum()
{
   CVisBefDoc *pDoc = GetDoc();
	BOOL bWasanimating = pDoc->IsAnimating();

	if ( bWasanimating)
		pDoc->StopAnimating();

	CNumInput Dlg;
	Dlg.DoModal();

	if ( bWasanimating)
		pDoc->StartAnimating();
	
	return Dlg.m_input;
}


//////////////////////////////////////////////////////////
// Called when a grid element has changed.
//
void CellChanged(long x, long y, char thischar, BOOL bIsPC )
{
   CVisBefDoc *pDoc = GetDoc();
	pDoc->CheckWatch(CPoint(x,y));

	CVisBefView *pView = GetView();
	pView->DrawDirect(x, y, thischar, bIsPC);
}


//////////////////////////////////////////////////////////
// Print integer value in output window.
//
void OutputInt( long v)
{
	CEdit *pWnd = GetOutputWindow();
	ASSERT(pWnd);

	CString str;
	str.Format("%ld ",v);	//Extra space is intentional

	//Paste given value to end of edit control
	int start, end;
	pWnd->SetSel( 0, -1) ; 
	pWnd->GetSel(start, end);
	pWnd->SetSel(end,end,TRUE);
	pWnd->ReplaceSel(str);
}


//////////////////////////////////////////////////////////
// Print single character in output window.
//
void OutputChar( char c)
{
	CEdit *pWnd = GetOutputWindow();
	ASSERT(pWnd);

	CString str;
	if ( c == '\n' )
		str = CString("\r\n");
	else
		str.Format("%c",c);

	//Paste given value to end of edit control
	int start, end;
	pWnd->SetSel( 0, -1) ; 
	pWnd->GetSel(start, end);
	pWnd->SetSel(end,end,TRUE);
	pWnd->ReplaceSel(str);
}


//////////////////////////////////////////////////////////
// Print a string in output window.
//
// Note: string output is not really a feature of befunge,
//       but it's quite handy for status messages.
//
void OutputString( LPCSTR str)
{
	CEdit *pWnd = GetOutputWindow();
	ASSERT(pWnd);

	//Paste given value to end of edit control
	int start, end;
	pWnd->SetSel( 0, -1) ; 
	pWnd->GetSel(start, end);
	pWnd->SetSel(end,end,TRUE);
	pWnd->ReplaceSel(str);
}


////////////////////////////////////////////////////////////
//This function is called when a value has been pushed on the
// stack.
//
void PushStack( long v)
{
	CListBox *pWnd = GetStackWindow();

   CString str;
	if ( v >= 32 && v < 128 )
	{
		//Show the corresponding ASCII char as well
		str.Format("%03ld - \'%c\'",v, (char) v);
	}
	else
	{
		str.Format("%ld",v);
	}
	pWnd->InsertString(0,str);

}


////////////////////////////////////////////////////////////
// This function is called when a value has been popped of
// the stack.
//
void PopStack()
{
	CListBox *pWnd = GetStackWindow();

	//Ignore any errors
	pWnd->DeleteString(0);
}


void ClearStack()
{
	CListBox *pWnd = GetStackWindow();
	if  (pWnd) 
		pWnd->ResetContent();
}
