///////////////////////////////////////////////////////////////
// DATE       : 15-5-97
// AUTHOR     : Wim Rijnders
// LAST_UPDATE: 1-7-97
// TITLE      : C++ implementation Visual Befunge interpreter.
// PROJECT    : VISBEF
// VERSION    : 1
// PLATFORM   : WIN32,, MFC
//-------------------------------------------------------------
// DESCRIPTION
// ===========
//
// 'Visually enhanced' C++ implementation of befung '93.
//
// Based on the befunge '93 befunge by Chris Pressey, the deranged 
// genius who thought up this language. It's still Bef'93, but with
// anticipated support of  features that will probably make it
// to the Bef'97 specs. 
//-------------------------------------------------------------
// NOTES
// =====
//
// * I honestly tried to keep it C-ish but the complexity of it
//   all went over my head. I gave up and switched to C++.
//
// * Support for multithreading/multitasking and gigantic befunge
//   spaces has been built in, but is not really used yet.
//
// * v10err_compat has been dumped from this version of befunge; 
//   we don't really need to be compatible with bef'93 since 
//   bef'96 (or earlier) compatibility is not our ambition anyway.
//
//   However, as a matter of assigning credit, this is the 
//   original comment Chris associated with this variable:
//
// 	!* as pointed out by Matthew D Moss.  *!
//	   !* backwards compatability for the    *!
//	   !* off-by-one error is only available *!
//	   !* in the interpreter... sorry!       *!
//
//-------------------------------------------------------------
// HISTORY
// =======
//
// 15- 5-97 Wim Rijnders v1: Created it.
///////////////////////////////////////////////////////////////
#include "stdafx.h"
#include <stdio.h>	
#include "befclass.h"


///////////////////////////////////////////////////////////////
// Class CBefThread
///////////////////////////////////////////////////////////////


const GridType CBefThread::getcur() 
{ 
	return CBefunge::GetCur(x,y); 
}


void CBefThread::setcur(GridType val) 
{ 
	CBefunge::SetCur(x,y, val); 
}


///////////////////////////////////////////////////////////////
// Perform a single PC step.
// (Chris Pressey should recognize this bit )
//
// Returns 1 if PC terminated.
//
StepResult CBefThread::Step()
{
   //PC is going to move - flag grid element as changed
	::CellChanged(x, y,getcur(),0);

	if ((getcur() != '@') || (stringmode))          /*** Intepreting Phase */
	{
		if (stringmode && (getcur() != '"'))
			push (getcur() );
		else 
		if (isdigit (getcur()))
			push (getcur() - '0');
		else
		switch (getcur())
		{
			case '>':            /* PC Right */
				dx = 1;
				dy = 0;
				break;
			case '<':            /* PC Left */
				dx = -1;
				dy = 0;
				break;
			case '^':            /* PC Up */
				dx = 0;
				dy = -1;
				break;
			case 'v':            /* PC Down */
				dx = 0;
				dy = 1;
				break;
			case '|':            /* Vertical 'If' */
				dx = 0;
				if (pop ())
					dy = -1;
				else
					dy = 1;
				break;
			case '_':            /* Horizontal 'If' */
			   dy = 0;
				if (pop ())
					dx = -1;
				else
					dx = 1;
				break;
			case '+':            /* Add */
				push (pop () + pop ());
				break;
			case '-':            /* Subtract */
			{
				long a = pop();
				long b = pop();
				push(b - a);
			}
			break;
			case '*':            /* Multiply */
				push (pop () * pop ());
				break;
			case '/':            /* Integer Divide */
			{
				signed long a = pop ();
				signed long b = pop ();

				/* WR8-5-97 Division by zero error spotted by Denis Moskovitz */
				if ( a== 0 )
				{
					push(0xFFFFFFFFL);
					OutputString("\r\n[Division by Zero!]\r\n");
				}
				else
					push (b / a);
			}
			break;
			case '%':            /* Modulo */
			{
				signed long a = pop ();
				signed long b = pop ();
				push (b % a);
			}
			break;
			case '\\':           /* Swap */
			{
				signed long a = pop ();
				signed long b = pop ();
				push (a);
				push (b);
			}
			break;
			case '.':            /* Pop Out Integer */
				OutputInt(pop() );
				break;
			case ',':            /* Pop Out ASCII */
				OutputChar( (char) pop() );
				break;
			case '"':		/* Toggle String Mode */
				stringmode = !stringmode;
				break;
			case ':':            /* Duplicate */
			{
				signed long a = pop ();
				push (a);
				push (a);
			}
			break;
			case '!':            /* Negate */
				if (pop())
					push(0);
				else
					push(1);
				break;
			case '`':
			{
				signed long b = pop ();
				signed long a = pop ();
				push (a > b);
			}
			break;
			case '#':            /* Bridge */
				x += dx;
				y += dy;
				break;
			case '$':            /* Pop and Discard */
				pop ();
				break;
			case '?':            /* Random Redirect */
				switch ((rand () / 32) % 4)
				{
				case 0:
					dx = 1;
					dy = 0;
					break;
				case 1:
					dx = -1;
					dy = 0;
					break;
				case 2:
					dx = 0;
					dy = -1;
					break;
				case 3:
					dx = 0;
					dy = 1;
					break;
				}
				break;
			case '&':            /* Input Integer */
			{
				signed long b = GetInputNum();
				push (b);
			}
			break;
			case '~':            /* Input ASCII */
			{
				char c = GetInputChar();
				push (c);
			}
			break;
			case 'g':            /* Get Value */
			{
				signed long y = pop (), x = pop ();
//				push (cur());
				push(CBefunge::GetCur(x,y));
			}
			break;
			case 'p':            /* Put Value */
			{
				signed long y = pop (), x = pop ();
//				cur() = (char) pop ();
				CBefunge::SetCur(x,y, (GridType) pop () );

				//Flag put-position as changed
				::CellChanged(x, y, CBefunge::GetCur(x,y),0);
			}
			break;
			default:
				break;
		}
		x += dx;
		y += dy;
		if (x < 0)
			x = CBefWorld::GetWidth() - 1;
		else
			x = x % CBefWorld::GetWidth();
		if ( y < 0)
			y = CBefWorld::GetHeight() - 1;
		else
			y = y % CBefWorld::GetHeight();

	   //Flag current position as changed
		::CellChanged(x, y,getcur(),1);
		return BEFOK;
	}

	return TERMINATED;
}


void CBefThread::Init()
{
   x = 0;
	y = 0; 
   dx = 1;
	dy = 0;
   stringmode = 0;
   ClearStack();
}


///////////////////////////////////////////////////////////////
// Class CBefTask
///////////////////////////////////////////////////////////////


CBefTask &CBefTask::operator=(CBefTask &task) 
{ 
	m_curthread = task.m_curthread; 

	m_thread.RemoveAll();
	for ( int i =0; i < task.m_thread.GetSize(); i ++ )
		m_thread.Add(task.m_thread[i]);

	return *this;
}


///////////////////////////////////////////////////////////////
// A task step consists of a single step of the next available 
// PC (or thread).
//
StepResult CBefTask::Step()
{
	StepResult tempresult;
	BOOL bHitBreak = FALSE;

	//
	//Following loop skips spaces
   //
	int numspaces = 0;

	do 
	{
		tempresult = m_thread[m_curthread].Step();
	} while (tempresult != TERMINATED &&
			  CBefunge::IsWarping() &&
		     m_thread[m_curthread].getcur() == ' ' &&
           ++numspaces < CBefWorld::GetWidth() &&
			  !CBefunge::HitBreakPoint(m_thread[m_curthread].GetPos() ) );
				 

	bHitBreak = CBefunge::HitBreakPoint( m_thread[m_curthread].GetPos() );

	if ( tempresult == TERMINATED )
		m_thread.RemoveAt(m_curthread);
	else 
	if ( numspaces == CBefWorld::GetWidth())
	{
		m_thread.RemoveAt(m_curthread);
		OutputString("[Thread has entered a Zero-Time Space Warp Continuum - bailing out]\r\n");
	}
	else
		m_curthread = ( m_curthread + 1 ) % m_thread.GetSize();

   //Termination of a task takes precedence over a breakpoint
	if ( m_thread.GetSize() == 0 )
		return TERMINATED;

	if ( bHitBreak )
		return BREAKPOINT;

	return BEFOK;
}


///////////////////////////////////////////////////////////////
// Class CBefunge
///////////////////////////////////////////////////////////////


//Static declaration(s)
CBefWorld CBefunge::m_world;
BOOL CBefunge::m_warpspace = TRUE;
CArray<CPoint, CPoint &> CBefunge::m_breakpoint;


///////////////////////////////////////////////////////////////
// Returns index of given breakpoint, or -1 if not found
//
int CBefunge::FindBreakPoint( CPoint &pos)
{
	for ( int i = 0; i < m_breakpoint.GetSize(); ++ i)
		if ( m_breakpoint[i] == pos )
		   return i;

	return -1;
}


BOOL CBefunge::HitBreakPoint( CPoint &pos)
{
	return FindBreakPoint( pos) != -1;
}


void CBefunge::ToggleBreakPoint( CPoint &pos)
{
	int index = FindBreakPoint(pos);

	if ( index != -1 )
		m_breakpoint.RemoveAt(index);
	else
		m_breakpoint.Add(pos);
}


void CBefunge::loadfile(FILE *f)
{
	int x = 0, y = 0, inp;
	
	while ( ( inp = fgetc(f) ) != EOF )
	{
		if (inp  == '\n')
		{
			SetCur(x,y, ' ');
			x = 0;
			y++;
			if (y >= CBefWorld::GetHeight())
				break;
		}
		else
		{
			SetCur(x,y,inp);
			x++;
			if (x >= CBefWorld::GetWidth())
			{
				x = 0;
				y++;
				if (y >= CBefWorld::GetHeight())
					break;
			}
		}
	}
}


///////////////////////////////////////////////////////////////
// A befunge step consists of a single step of every running 
// task.
//
StepResult CBefunge::Step()
{
	StepResult tempresult;
	BOOL bHitBreak = FALSE;

	int i = 0;
	while ( i < m_task.GetSize() )
	{
		tempresult = m_task[i].Step();

	   bHitBreak |= ( tempresult == BREAKPOINT );

		if ( tempresult == TERMINATED )
			m_task.RemoveAt(i);
		else
			++i;
	}

	if ( m_task.GetSize() == 0 )
		return TERMINATED;
	else
	if ( bHitBreak )
	   return BREAKPOINT;
	else
		return BEFOK;
	
}


///////////////////////////////////////////////////////////////
// Determine last non-space char on line with given index;
//
// returns: -1 if line entirely empty.
//
// This routine is only used in conjunction with savefile
//
int CBefunge::ScanLastNonSpace(int y)
{
	ASSERT ( y >= 0 && y < m_world.GetWidth() );

	int x = m_world.GetWidth() - 1;

	while ( x >= 0 && GetCur(x,y) == ' ') 
		x--;

	return x;
}


void CBefunge::savefile(FILE *f)
{
	int lastx,x,y;
	//Search for last non-empty line
	int lasty = m_world.GetHeight() - 1; 

	while ( lasty > 0 && ScanLastNonSpace(lasty)== -1 )
		lasty--;
	
	for ( y = 0; y <= lasty; ++ y)
	{
		lastx = ScanLastNonSpace(y);

		for ( x = 0; x <= lastx; ++ x)
			fputc( GetCur(x,y), f );

		fputc( '\n' , f );
	}
}
