
#include <stdlib.h>
#include <string.h>
#include "assert.h"
#include "types.h"

/*
   Define FAST_CUTOUTS for a vast speedup using a lookup table.
   I've left the old version in because it's easier to prove correct.
*/
#define FAST_CUTOUTS 1


board *new_board(int w, int h)
{
    board *b = malloc(sizeof *b);
    assert(b);
    b->turn = PLAYER1;
    b->w = w; b->h = h;
    assert(w > 0 && h > 0);
    b->data = malloc(w*h * sizeof *b->data);
    assert(b->data);
    memset(b->data, NONE, w*h);
#define init(i, let)                      \
    b->pieces[i].letter = let;            \
    b->pieces[i].x = b->pieces[i].y = -1; \
    b->pieces[i].rot = 0;                 \
    b->pieces[i].owner = UNOWNED;

    init(0, 'F');
    init(1, 'I');
    init(2, 'L');
    init(3, 'N');
    init(4, 'P');
    init(5, 'T');
    init(6, 'U');
    init(7, 'V');
    init(8, 'W');
    init(9, 'X');
    init(10, 'Y');
    init(11, 'Z');
    return b;
}

board *copy_board(const board *b)
{
    board *b2 = malloc(sizeof *b2);
    assert(b2);
    memcpy(b2, b, sizeof *b);
    b2->data = malloc(b->w * b->h * sizeof *b->data);
    assert(b2->data);
    memcpy(b2->data, b->data, b->w * b->h * sizeof *b->data);
    return b2;
}

void kill_board(board *b)
{
    free(b->data);
    free(b);
}


piece *piece_of_letter(const board *b, int letter)
{
    char *names = "FILNPTUVWXYZ";
    int which = strchr(names, letter) - names;
    return (piece *)&b->pieces[which];
}

piece *piece_of_position(const board *b, int x, int y)
{
    int who = b->data[y*b->w+x];
    int i, m, n;
    if (who == NONE) return NULL;
    for (i=0; i < NELEM(b->pieces); ++i) {
        cutout *c;
        if (b->pieces[i].owner != who) continue;
        if (b->pieces[i].x < 0) continue;
        m = x - b->pieces[i].x;
        n = y - b->pieces[i].y;
        if (m < 0 || n < 0 || m >= 5 || n >= 5) continue;
        c = cutout_of_piece(&b->pieces[i]);
        assert(c);
        if (m < c->w && n < c->h && c->data[n*c->w+m] == FILLED) {
            kill_cutout(c);
            return (piece *)&b->pieces[i];
        }
        kill_cutout(c);
    }
    assert(!"piece_of_position failed");
    return NULL;
}


cutout *cutout_of_piece(const piece *p)
{
    return cutout_of_letter(p->letter, p->rot);
}

#if FAST_CUTOUTS

static cutout all_cutouts[] = {
    {3,3, ".FFFF..F."},
    {3,3, "F..FFF.F."},
    {3,3, ".F..FFFF."},
    {3,3, ".F.FFF..F"},
    {3,3, "FF..FF.F."},
    {3,3, ".F.FFFF.."},
    {3,3, ".F.FF..FF"},
    {3,3, "..FFFF.F."},
    {1,5, "IIIII"},
    {5,1, "IIIII"},
    {2,4, "L.L.L.LL"},
    {4,2, "...LLLLL"},
    {2,4, "LL.L.L.L"},
    {4,2, "LLLLL..."},
    {2,4, ".L.L.LLL"},
    {4,2, "LLLL...L"},
    {2,4, "LLL.L.L."},
    {4,2, "L...LLLL"},
    {2,4, ".NNNN.N."},
    {4,2, "NN...NNN"},
    {2,4, ".N.NNNN."},
    {4,2, "NNN...NN"},
    {2,4, "N.NN.N.N"},
    {4,2, ".NNNNN.."},
    {2,4, "N.N.NN.N"},
    {4,2, "..NNNNN."},
    {2,3, "PPPPP."},
    {3,2, "PP.PPP"},
    {2,3, ".PPPPP"},
    {3,2, "PPP.PP"},
    {2,3, "PPPP.P"},
    {3,2, "PPPPP."},
    {2,3, "P.PPPP"},
    {3,2, ".PPPPP"},
    {3,3, "TTT.T..T."},
    {3,3, "T..TTTT.."},
    {3,3, ".T..T.TTT"},
    {3,3, "..TTTT..T"},
    {3,2, "U.UUUU"},
    {2,3, "UU.UUU"},
    {3,2, "UUUU.U"},
    {2,3, "UUU.UU"},
    {3,3, "V..V..VVV"},
    {3,3, "..V..VVVV"},
    {3,3, "VVV..V..V"},
    {3,3, "VVVV..V.."},
    {3,3, "..V..VVVV"},
    {3,3, "VVV..V..V"},
    {3,3, "VVVV..V.."},
    {3,3, "V..V..VVV"},
    {3,3, "W..WW..WW"},
    {3,3, "..W.WWWW."},
    {3,3, "WW..WW..W"},
    {3,3, ".WWWW.W.."},
    {3,3, "..W.WWWW."},
    {3,3, "WW..WW..W"},
    {3,3, ".WWWW.W.."},
    {3,3, "W..WW..WW"},
    {3,3, ".X.XXX.X."},
    {2,4, ".YYY.Y.Y"},
    {4,2, "YYYY.Y.."},
    {2,4, "Y.Y.YYY."},
    {4,2, "..Y.YYYY"},
    {2,4, "Y.YYY.Y."},
    {4,2, ".Y..YYYY"},
    {2,4, ".Y.YYY.Y"},
    {4,2, "YYYY..Y."},
    {3,3, "ZZ..Z..ZZ"},
    {3,3, "..ZZZZZ.."},
    {3,3, "ZZ..Z..ZZ"},
    {3,3, "..ZZZZZ.."},
    {3,3, ".ZZ.Z.ZZ."},
    {3,3, "Z..ZZZ..Z"},
    {3,3, ".ZZ.Z.ZZ."},
    {3,3, "Z..ZZZ..Z"},
};

cutout *cutout_of_letter(int letter, int rot)
{
    if (all_cutouts[0].data[0] == '.') {
        int i, j;
        /* First time through, change ('x', '.') to (FILLED, NONE). */
        for (i=0; i < NELEM(all_cutouts); ++i) {
            for (j=0; j < all_cutouts[i].w * all_cutouts[i].h; ++j) {
                all_cutouts[i].data[j] =
                    (all_cutouts[i].data[j]=='.')? NONE: FILLED;
            }
        }
    }

    switch (letter) {
        case 'F': return &all_cutouts[rot   + 0];
        case 'I': return &all_cutouts[rot%2 + 8];
        case 'L': return &all_cutouts[rot   + 10];
        case 'N': return &all_cutouts[rot   + 18];
        case 'P': return &all_cutouts[rot   + 26];
        case 'T': return &all_cutouts[rot%4 + 34];
        case 'U': return &all_cutouts[rot%4 + 38];
        case 'V': return &all_cutouts[rot   + 42];
        case 'W': return &all_cutouts[rot   + 50];
        case 'X': return &all_cutouts[0     + 58];
        case 'Y': return &all_cutouts[rot   + 59];
        case 'Z': return &all_cutouts[rot   + 67];
    }
    /* NOT REACHED */
    assert(!"unmatched letter in cutout_of_letter");
    return NULL;
}

void kill_cutout(cutout *c) { (void)c; }

#else /* FAST_CUTOUTS */

cutout *cutout_of_letter(int letter, int rot)
{
    cutout *c = malloc(sizeof *c);
    char *names = "FILNPTUVWXYZ";
    int which = strchr(names, letter) - names;
    int i, j;
    const char *maps[12] = {
        ".xx.."
        "xx..."
        ".x..."
        "....."
        ".....",
        "x...."
        "x...."
        "x...."
        "x...."
        "x....",
        "x...."
        "x...."
        "x...."
        "xx..."
        ".....",
        ".x..."
        "xx..."
        "x...."
        "x...."
        ".....",
        "xx..."
        "xx..."
        "x...."
        "....."
        ".....",
        "xxx.."
        ".x..."
        ".x..."
        "....."
        ".....",
        "x.x.."
        "xxx.."
        "....."
        "....."
        ".....",
        "x...."
        "x...."
        "xxx.."
        "....."
        ".....",
        "x...."
        "xx..."
        ".xx.."
        "....."
        ".....",
        ".x..."
        "xxx.."
        ".x..."
        "....."
        ".....",
        ".x..."
        "xx..."
        ".x..."
        ".x..."
        ".....",
        "xx..."
        ".x..."
        ".xx.."
        "....."
        "....."
    };

    /* Copy the image of the piece in orientation 0. */
    assert(c);
    c->w = c->h = 5;
    memcpy(c->data, maps[which], 25);

    /* Rotate the piece to its proper orientation. */
#define DATA(y,x) c->data[5*(y)+(x)]
#define SWAP(ay,ax, by,bx) do {    \
        int ttmp = DATA(by,bx);    \
        DATA(by,bx) = DATA(ay,ax); \
        DATA(ay,ax) = ttmp;        \
    } while (0)
#define CYCLE(ay,ax, by,bx, cy,cx, dy,dx) do { \
        int ttmp = DATA(dy,dx);                \
        DATA(dy,dx) = DATA(cy,cx);             \
        DATA(cy,cx) = DATA(by,bx);             \
        DATA(by,bx) = DATA(ay,ax);             \
        DATA(ay,ax) = ttmp;                    \
    } while (0)

    if (rot >= 4) {
        /* Flip horizontal. */
        for (i=0; i < 5; ++i) {
            SWAP(i,0, i,4);
            SWAP(i,1, i,3);
        }
        rot -= 4;
    }
    if (rot >= 2) {
        /* Rotate 180 degrees. */
        for (i=0; i < 2; ++i) {
            SWAP(i,0, (4-i),4);
            SWAP(i,1, (4-i),3);
            SWAP(i,2, (4-i),2);
            SWAP(i,3, (4-i),1);
            SWAP(i,4, (4-i),0);
        }
        SWAP(2,0, 2,4);
        SWAP(2,1, 2,3);
        rot -= 2;
    }
    if (rot >= 1) {
        /* Rotate 90 degrees left. */
        CYCLE(0,0, 4,0, 4,4, 0,4);
        CYCLE(0,1, 3,0, 4,3, 1,4);
        CYCLE(0,2, 2,0, 4,2, 2,4);
        CYCLE(0,3, 1,0, 4,1, 3,4);
        CYCLE(1,1, 3,1, 3,3, 1,3);
        CYCLE(1,2, 2,1, 3,2, 2,3);
        rot -= 1;
    }
    assert(rot == 0);

    for (i=0; i < 4; ++i) {
        int okay = 0;
        for (j=0; j < 5; ++j)
          if (DATA(j,0) == 'x') okay = 1;
        if (okay) break;
        /* Shift everything left one row. */
        for (j=0; j < 5; ++j) {
            DATA(j,0) = DATA(j,1);
            DATA(j,1) = DATA(j,2);
            DATA(j,2) = DATA(j,3);
            DATA(j,3) = DATA(j,4);
            DATA(j,4) = '.';
        }
    }

    for (i=0; i < 4; ++i) {
        int okay = 0;
        for (j=0; j < 5; ++j)
          if (DATA(0,j) == 'x') okay = 1;
        if (okay) break;
        /* Shift everything up one row. */
        for (j=0; j < 5; ++j) {
            DATA(0,j) = DATA(1,j);
            DATA(1,j) = DATA(2,j);
            DATA(2,j) = DATA(3,j);
            DATA(3,j) = DATA(4,j);
            DATA(4,j) = '.';
        }
    }

    /* Convert the data from ('x', '.') to (FILLED, NONE). */
    for (i=0; i < c->w*c->h; ++i)
      c->data[i] = (c->data[i]=='x')? FILLED: NONE;

    return c;
}
#undef DATA
#undef SWAP
#undef CYCLE


void kill_cutout(cutout *c)
{
    free(c);
}

#endif /* FAST_CUTOUTS */


int fits(const board *b, const cutout *c, int x, int y)
{
    int i, j;
    assert(x >= 0 && y >= 0);
    for (j=0; j < c->h; ++j) {
        for (i=0; i < c->w; ++i) {
            if (c->data[j*c->w+i] != FILLED)
              continue;
            if (x+i >= b->w) return 0;
            if (y+j >= b->h) return 0;
            if (b->data[(y+j)*b->w+(x+i)] != NONE) return 0;
        }
    }
    return 1;
}

/* place(b,c,x,y, NONE) removes a piece from the board */
void place(board *b, const cutout *c, int x, int y, int who)
{
    int i, j;
    assert(x >= 0 && y >= 0);
    for (j=0; j < c->h; ++j) {
        for (i=0; i < c->w; ++i) {
            if (c->data[j*c->w+i] != FILLED)
              continue;
            assert(x+i < b->w && y+j < b->h);
            b->data[(y+j)*b->w+(x+i)] = who;
        }
    }
}

void place_piece(board *b, const piece *p)
{
    cutout *c = cutout_of_piece(p);
    place(b, c, p->x, p->y, p->owner);
    kill_cutout(c);
}
