
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "assert.h"
#include "types.h"
#include "moves.h"
#include "alpha.h"
#include "display.h"

#define ABS(x) ((x) >= 0? (x): -(x))

static int equiv_rots(int letter, int r1, int r2);
static int easily_confused_rots(int r1, int r2);
static int confirmed(const char *text);
static possible_move *convert_to_move(const char *text, const board *b);


static void
display_board_only(const board *b, int indent)
{
    int i, j;
    for (j=0; j < b->h; ++j) {
        printf("%-*s", indent, "");
        for (i=0; i < b->w; ++i) {
            const piece *p = piece_of_position(b, i, j);
            if (p == NULL) printf(".");
            else {
                if (p->owner == b->turn)
                  printf("%c", p->letter);
                else printf("%c", tolower(p->letter));
            }
        }
        printf("\n");
    }
}


void display_board(const char *msg, const board *b)
{
    int i;
    int otherguy = (b->turn == PLAYER1)? PLAYER2: PLAYER1;
    int oppo, avail, off;

    printf("    %s\n\n", msg);
    display_board_only(b, 4);

    oppo = avail = off = 0;
    for (i=0; i < NELEM(b->pieces); ++i) {
        if (b->pieces[i].x < 0) {
            if (b->pieces[i].owner == UNOWNED) ++avail;
            else ++off;
        }
        else if (b->pieces[i].owner == otherguy) ++oppo;
    }

    printf("\n    Available:");
    for (i=0; i < NELEM(b->pieces); ++i)
      if (b->pieces[i].x < 0 && b->pieces[i].owner == UNOWNED)
        printf(" %c", b->pieces[i].letter);
    if (off > 0) {
        printf("\n    Out of circulation:");
        for (i=0; i < NELEM(b->pieces); ++i)
          if (b->pieces[i].x < 0 && b->pieces[i].owner != UNOWNED)
            printf(" %c", b->pieces[i].letter);
    }
    if (oppo > 0) {
        printf("\n    Player %d owns:", (b->turn==PLAYER1)? 2: 1);
        for (i=0; i < NELEM(b->pieces); ++i)
          if (b->pieces[i].x >= 0 && b->pieces[i].owner == otherguy)
            printf(" %c", b->pieces[i].letter);
    }
    printf("\n\n");
}

possible_move *get_users_move(const board *b)
{
    static int first = 1;
    char buffer[100];
    possible_move *m;

    puts("Enter your move, e.g., W15; P; F33 R3.");
    if (first) {
        puts("The first letter gives the name of the piece.");
        puts("Optionally, two numbers give top-left row and column.");
        puts("Optionally, \"R\" plus number gives rotation.");
        puts("Your move will be displayed for confirmation.");
        puts("Confirm with \"C\", or enter a new move.");
        puts("To remove one of your pieces, just enter its letter.");
        puts("Removals are not confirmed.");
        first = 0;
    }

    while (1) {
        assert(fgets(buffer, sizeof buffer, stdin) != NULL);
      was_it_a_move:
        m = convert_to_move(buffer, b);

        if (m == NULL) {
            puts("That move didn't parse or wasn't valid; try again.");
            continue;
        }

        if (m->x < 0) break;  /* accept removals without confirm */
        else {
            board *b2 = copy_board(b);
            make_move(b2, m->letter, m->rot, m->x, m->y);
            b2->turn = b->turn;
            if (m->value < 50)
              puts("This is the move I think you mean:");
            else puts("This is the move you selected:");
            display_board_only(b2, 2);
            kill_board(b2);
            puts("Enter C to confirm.");
            assert(fgets(buffer, sizeof buffer, stdin) != NULL);
            if (confirmed(buffer)) break;
            free(m);
            goto was_it_a_move;
        }
    }

    return m;
}


static int
confirmed(const char *text)
{
    int i = 0;
    while (text[i] && isspace(text[i])) ++i;
    return (text[i] && toupper(text[i]) == 'C');
}

static possible_move *
convert_to_move(const char *text, const board *b)
{
    int i = 0;
    int letter, rot = -1, x = -1, y = -1;
    possible_move *m, *best;

    while (text[i] && isspace(text[i])) ++i;
    letter = toupper(text[i]);
    if (letter == '\0' || !strchr("FILNPTUVWXYZ", letter))
      return NULL;

  keep_going:
    do { ++i; } while (text[i] && isspace(text[i]));
    if (toupper(text[i]) == 'R') {
        if (rot != -1) return NULL;
        do { ++i; } while (text[i] && isspace(text[i]));
        if (!isdigit(text[i])) return NULL;
        rot = text[i] - '0';
        if (rot > 7) return NULL;
        goto keep_going;
    }
    else if (isdigit(text[i])) {
        if (x != -1) return NULL;
        y = text[i] - '0';
        do { ++i; } while (text[i] && isspace(text[i]));
        if (!isdigit(text[i])) return NULL;
        x = text[i] - '0';
        goto keep_going;
    }
    else if (text[i] != '\0') return NULL;

    /* At this point we have parsed the line completely. */
    m = find_all_possible_moves(b);
    best = NULL;
    while (m != NULL) {
        board *b2;
        int this_e = 0;
        possible_move *cur = m;
        /* Is this a possible move? Is it better than the one we have? */
        if (m->letter != letter) goto reject;
        if (rot == -1 || equiv_rots(letter, m->rot, rot))
          this_e += 50;
        else if (easily_confused_rots(m->rot, rot))
          this_e += 25;
        else goto reject;
        if (x == -1) this_e -= 0;
        else if (ABS(m->x - x) > 2) goto reject;
        else if (ABS(m->y - y) > 2) goto reject;
        else this_e -= 3*(ABS(m->x - x) + ABS(m->y - y));

        b2 = copy_board(b);
        make_move(b2, m->letter, m->rot, m->x, m->y);
        b2->turn = PLAYER1;
        this_e += evaluate_ownership(b2);
        if (best == NULL || this_e > best->value) {
            best = cur;
            best->value = this_e;
            cur = NULL;
        }
      reject:
        m = m->next;
        free(cur);
    }
    return best;
}


static int
equiv_rots(int letter, int r1, int r2)
{
    if (r1 == r2) return 1;
    switch (letter) {
        case 'F': return 0;
        case 'I': return (r1 % 2 == r2 % 2);
        case 'L': return 0;
        case 'N': return 0;
        case 'P': return 0;
        case 'T': return (r1 % 4 == r2 % 4);
        case 'U': return (r1 % 4 == r2 % 4);
        case 'V': case 'W': do {
            if (r1 == 7) r1 = 0;
            if (r2 == 7) r2 = 0;
            if (r1 >= 4) r1 -= 3;
            if (r2 >= 4) r2 -= 3;
            return (r1 == r2);
        } while (0);
        case 'X': return 1;
        case 'Y': return 0;
        case 'Z': return (r1 % 2 == r2 % 2) && ((r1 < 4) == (r2 < 4));
    }
    assert(!"Unmatched letter in rot_duplicates");
    return 0; /* NOT REACHED */
}


static int easily_confused_rots(int r1, int r2)
{
    switch (r1) {
        case 0: case 4: return (r2 == 0 || r2 == 4);
        case 1: case 3: case 5: case 7: return (r2 % 2 == 1);
        case 2: case 6: return (r2 == 2 || r2 == 6);
    }
    assert(!"unmatched r1 in easily_confused_rots");
    return 0; /* NOT REACHED */
}
