
#include <stdlib.h>
#include "assert.h"
#include "types.h"
#include "moves.h"
#include "alpha.h"

static possible_move *
pmove(int let, int rot, int x, int y, possible_move *next)
{
    possible_move *p = malloc(sizeof *p);
    assert(p);
    p->letter = let; p->rot = rot;
    p->x = x; p->y = y;
    p->value = 0;
    p->next = next;
    return p;
}

/* Return 1 if this rotation is equivalent to any lesser rotation. */
static int
rot_duplicates(int letter, int rot)
{
    switch (letter) {
        case 'F': return 0;
        case 'I': return !(rot == 0 || rot == 1);
        case 'L': return 0;
        case 'N': return 0;
        case 'P': return 0;
        case 'T': return (rot > 3);
        case 'U': return (rot > 3);
        case 'V': return (rot > 3);
        case 'W': return (rot > 3);
        case 'X': return (rot != 0);
        case 'Y': return 0;
        case 'Z': return !(rot == 0 || rot == 1 || rot == 4 || rot == 5);
    }
    assert(!"Unmatched letter in rot_duplicates");
    return 0; /* NOT REACHED */
}


possible_move *find_all_possible_moves(const board *b)
{
    possible_move *head = NULL;
    const char *names = "FILNPTUVWXYZ";
    int which;
    int x, y;
    int rot;

    for (which=0; names[which] != '\0'; ++which) {
        const piece *p = &b->pieces[which];
        if (p->x >= 0) {
            /* Remove the piece. */
            if (p->owner == b->turn)
              head = pmove(names[which], 0, -1, -1, head);
        }
        else {
            /* Put the piece on the board somewhere. */
            if (p->owner != UNOWNED) continue;
            for (rot=0; rot < 8; ++rot) {
                if (rot_duplicates(names[which], rot)) continue;
                for (x=0; x < b->w; ++x)
                  for (y=0; y < b->h; ++y)
                    if (move_possible(b, names[which], rot, x, y))
                      head = pmove(names[which], rot, x, y, head);
            }
        }
    }
    return head;
}


/* Return a positive number if the board is owned mostly by b->turn. */
int evaluate_ownership(const board *b)
{
    int i;
    int sum = 0;
    for (i=0; i < NELEM(b->pieces); ++i) {
        if (b->pieces[i].x >= 0) {
            sum += (b->pieces[i].owner == PLAYER1);
            sum -= (b->pieces[i].owner == PLAYER2);
        }
    }
    return (b->turn == PLAYER1)? sum: -sum;
}


/* Return the alpha-beta value of the given board for b->turn. */
possible_move *eval_alpha_beta_depth(const board *b, int depth)
{
    possible_move *poss, *best = NULL;
    poss = find_all_possible_moves(b);
    /* Note that having no moves means you lose! */

    while (poss != NULL) {
        possible_move *cur = poss;
        board *b2 = copy_board(b);
        int this_e;
        make_move(b2, cur->letter, cur->rot, cur->x, cur->y);
        if (depth > 0) {
            possible_move *m = eval_alpha_beta_depth(b2, depth-1);
            this_e = (m == NULL)? -1000: m->value;
            if (m) free(m);
        }
        else this_e = evaluate_ownership(b2);

        /* If we just placed one, then this_e refers to the other guy! */
        if (cur->x >= 0) this_e = -this_e;

        if (best == NULL || this_e > best->value) {
            cur->value = this_e;
            if (best) free(best);
            /* Don't free the new best move, then. */
            best = cur;
            cur = NULL;
        }
        kill_board(b2);
        poss = poss->next;
        free(cur);
        /* Did we find a winning move yet? */
        if (best && best->value == 1000) break;
    }
    /* The rest of the moves are irrelevant; we've found a winner. */
    while (poss != NULL) {
        possible_move *cur = poss;
        poss = poss->next;
        free(cur);
    }

    return best;
}


possible_move *eval_alpha_beta_moves(const board *b, int moves)
{
    /* Stub. Todo fixme bug hack. */
    (void)b; (void)moves;
    return NULL;
}
